import {addFilter} from '@wordpress/hooks';
import {createHigherOrderComponent} from '@wordpress/compose';
import {Fragment, useRef, useEffect} from '@wordpress/element';
import {useDispatch} from '@wordpress/data';
import {store as noticeStore} from '@wordpress/notices';

const index = (BlockEdit) => {
    return createHigherOrderComponent((props) => {
        const ref = useRef();
        const {createWarningNotice} = useDispatch(noticeStore);

        useEffect(() => {
            const container = ref.current;
            if (!container) return;

            /**
             * Handle click events on the block editor.
             * If the click target is a link:
             *   1. Disable the link by setting aria-disabled to true
             *   2. Prevent the link from being followed
             *   3. Display a warning notice to the user
             * @param {MouseEvent} e - The click event.
             */
            const clickHandler = (e) => {
                const link = e.target.closest('a');

                if (link && container.contains(link) && !link.dataset.richTextFormatBoundary && !link.getAttribute('href').startsWith('#')) {
                    link.setAttribute('aria-disabled', 'true');
                    // Prevent the link from being followed
                    e.preventDefault();
                    // Display a warning notice to the user
                    createWarningNotice('Links are disabled in the editor.', {
                        type: 'snackbar',
                    });
                }

            };
            container.addEventListener('click', clickHandler);

        }, []);

        return (
            <Fragment>
                <div ref={ref}>
                    <BlockEdit {...props} />
                </div>
            </Fragment>
        );
    }, 'disableLinksInsideEditor');
};

addFilter(
    'editor.BlockEdit',
    'headless/aria-disabled-links',
    index
);