<?php

declare( strict_types=1 );

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Load the Headless theme.
 *
 * This file serves as the entry point for the theme, loading dependencies,
 * bootstrapping modules, and initializing the core system.
 */

require_once __DIR__ . '/vendor/autoload.php';
require_once __DIR__ . '/app/bootstrap.php';

define('RESOURCES_URL', get_stylesheet_directory_uri() . '/resources');

use Headless\Kernel\Core;
function d($value, $die = false) {
    \Headless\Utils\MyDebug::dump($value, $die);
}
/**
 * Theme setup and initialization.
 */
function quantum_theme_setup(): void {

	// Check if Core is available
	if ( ! class_exists( Core::class ) ) {
		error_log( 'Headless\Core class is missing. Check Composer autoload configuration.' );
		return;
	}

	// Theme-specific setup
	add_theme_support( 'post-thumbnails' );
	add_theme_support( 'title-tag' );
	add_theme_support( 'menus' );

	register_nav_menus( [
		'primary'   => __( 'Header menu', 'theme-textdomain' ),
		'footer'    => __( 'Footer Menu', 'theme-textdomain' )
	] );
}
add_action( 'after_setup_theme', 'quantum_theme_setup' );

// Speculation loading (WP 6.8+ only)
add_filter( 'wp_speculation_rules_configuration', function( $config ) {
    if ( ! is_array( $config ) ) {
        $config = [];
    }

    return array_merge( $config, [
        'eagerness' => 'moderate',
        'strategy'  => 'prerender',
    ]);
});


/**
 * AJAX handler for blog archive "Load more" button.
 */
function headless_load_more_posts() {
	check_ajax_referer( 'headless_load_more_posts', 'nonce' );

	// Base args from the current archive query.
	$query_vars = isset( $_POST['queryVars'] ) && is_array( $_POST['queryVars'] ) ? wp_unslash( $_POST['queryVars'] ) : [];

	$allowed_args = [
		'post_type',
		'posts_per_page',
		'orderby',
		'order',
		'cat',
		'category_name',
		'tag',
		'tax_query',
	];

	$args = array_intersect_key( $query_vars, array_flip( $allowed_args ) );

	$args['post_type']           = ! empty( $args['post_type'] ) ? $args['post_type'] : 'post';
	$args['posts_per_page']      = isset( $args['posts_per_page'] ) ? absint( $args['posts_per_page'] ) : get_option( 'posts_per_page' );
	$args['paged']               = isset( $_POST['page'] ) ? max( 1, absint( $_POST['page'] ) ) : 1;
	$args['post_status']         = 'publish';
	$args['ignore_sticky_posts'] = true;

	$query = new WP_Query( $args );

	ob_start();

	if ( $query->have_posts() ) {
		while ( $query->have_posts() ) {
			$query->the_post();
			get_template_part( 'template-parts/blog-card' );
		}
	}

	wp_reset_postdata();

	wp_send_json_success(
		[
			'html'    => ob_get_clean(),
			'maxPage' => (int) $query->max_num_pages,
		]
	);
}
add_action( 'wp_ajax_headless_load_more_posts', 'headless_load_more_posts' );
add_action( 'wp_ajax_nopriv_headless_load_more_posts', 'headless_load_more_posts' );

add_filter('excerpt_more', function($more){
    return '...';
});

add_filter( 'wp_speculation_rules_configuration', function( $config ) {
    if ( ! is_array( $config ) ) {
        $config = [];
    }

    return array_merge( $config, [
        'eagerness' => 'moderate',
        'strategy'  => 'prerender',
    ]);
});

add_filter( 'wp_speculation_rules_href_exclude_paths', function( $exclude ) {
    if ( ! is_array( $exclude ) ) {
        $exclude = [];
    }

    return array_merge( $exclude, [
        '/wp-login.php',
        '/wp-admin'
    ]);
});