<?php

namespace Headless\Modules\Menu;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
use Headless\Utils\CommonUtils;
use Headless\Interfaces\Registrable;
use Headless\Traits\Loadable;
use Walker_Nav_Menu;

class MenuWalker extends Walker_Nav_Menu implements Registrable {
	use Loadable;

	private array $submenu_counts = [];
	private array $menu_items_map = [];

	public function register(): void {
		add_filter( 'wp_nav_menu_args', [ $this, 'inject_walker' ] );
	}

	public function inject_walker( $args ) {
		if ( isset( $args['theme_location'] ) && $args['theme_location'] === 'primary' ) {
			$args['walker'] = $this;
		}
		return $args;
	}

	public function walk( $elements, $max_depth, ...$args ) {
		foreach ( $elements as $el ) {
			$this->menu_items_map[ $el->ID ] = $el;
			if ( $el->menu_item_parent ) {
				$parent_id = (int) $el->menu_item_parent;
				$this->submenu_counts[ $parent_id ] = ( $this->submenu_counts[ $parent_id ] ?? 0 ) + 1;
			}
		}
		return parent::walk( $elements, $max_depth, ...$args );
	}
	public function start_el( &$output, $item, $depth = 0, $args = null, $id = 0 ) {
		// Detect if item has children
		$is_parent = in_array(
			'menu-item-has-children',
			is_array( $item->classes ) ? $item->classes : [],
			true
		);
	
		$title = esc_html( $item->title );
		$url   = esc_url( $item->url );
	
		// ACF fields
		$text = get_field( 'menu-description', $item );
		$img  = get_field( 'menu-icon', $item );
	
		$has_acf = ! empty( $text ) || ! empty( $img );
	
		// ---------- <li> classes ----------
		// IMPORTANT: no js-openSubMenu / sub-menu-open here
		$li_classes = [ 'menu-item', 'd-flex', 'flex-md-row', 'flex-column' ];
	
		$output .= sprintf(
			'<li id="menu-item-%1$d" class="%2$s">',
			(int) $item->ID,
			esc_attr( implode( ' ', $li_classes ) )
		);
	
		// ---------- TOP LEVEL (depth 0) ----------
		if ( 0 === $depth ) {
	
			$link_classes = [];
			$class_attr   = ! empty( $link_classes ) ? ' class="' . esc_attr( implode( ' ', $link_classes ) ) . '"' : '';
	
			$description_html = $text
				? '<div class="description">' . esc_html( $text ) . '</div>'
				: '';
	
			$icon = CommonUtils::render_image_acf(
				$img,
				[
					'size'  => 'medium',
					'alt'   => 'Menu item ' . $item->ID . ' image',
					'class' => 'align-self-start',
				]
			);
	
			$output .= '<a href="' . $url . '"' . $class_attr . '>';
			$output .= $description_html;
			$output .= '<div class="text d-flex">' . $icon . $title;
	
			// js-openSubMenu / sub-menu-open ONLY on span with SVG
			if ( $is_parent ) {
				$output .= '<span class="js-openSubMenu sub-menu-open d-flex align-self-start">
					<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
						<path d="M6 9L12 15L18 9" stroke="#696C71" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
					</svg>
				</span>';
			}
	
			$output .= '</div></a>';
	
			return;
		}
	
		// ---------- LEVEL 1 ----------
		if ( 1 === $depth ) {
			$link_classes = [];
	
			if ( $is_parent ) {
				$link_classes[] = 'sub-menu-title';
			} else {
				$link_classes[] = 'sub-menu-link';
				$link_classes[] = 'd-block';
			}
	
			$class_attr = ' class="' . esc_attr( implode( ' ', $link_classes ) ) . '"';
	
			$is_hidden     = is_array( $item->classes ) && in_array( 'hidden-item', $item->classes, true );
			$tabindex_attr = $is_hidden ? ' tabindex="-1"' : ' tabindex="0"';
	
			$output .= '<a ' . $tabindex_attr . ' href="' . $url . '"' . $class_attr . '>' . $title . '</a>';
	
			return;
		}
	
		// ---------- LEVEL 2+ ----------
		if ( $depth >= 2 ) {
			$link_classes = [ 'sub-menu-link', 'd-block' ];
			$class_attr   = ' class="' . esc_attr( implode( ' ', $link_classes ) ) . '"';
			$output      .= '<a href="' . $url . '"' . $class_attr . '>' . $title . '</a>';
	
			return;
		}
	
		// Fallback
		$output .= '<a href="' . $url . '">' . $title . '</a>';
	}	
	

	public function start_lvl( &$output, $depth = 0, $args = null ) {
		$indent = str_repeat( "\t", $depth );
	
		// For first submenu level (children of top-level)
		if ( 0 === $depth ) {
			$output .= "\n$indent<div class=\"sub-menu-container\">\n";
			$output .= "$indent<ul class=\"sub-menu flex-column\">\n";
		} else {
			// If you ever have deeper levels – they will also be flex-column
			$output .= "\n$indent<ul class=\"sub-menu flex-column\">\n";
		}
	}
	
	public function end_lvl( &$output, $depth = 0, $args = null ) {
		$indent = str_repeat( "\t", $depth );
	
		if ( 0 === $depth ) {
			$output .= "$indent</ul>\n";
			$output .= "$indent</div>\n";
		} else {
			$output .= "$indent</ul>\n";
		}
	}
}