<?php

namespace Headless\Modules\Menu;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Headless\Interfaces\Registrable;
use Headless\Traits\Loadable;
use Walker_Nav_Menu;

class FooterMenuWalker extends Walker_Nav_Menu implements Registrable {
	use Loadable;

	/**
	 * ID of the last top-level (root) menu item.
	 *
	 * @var int
	 */
	private int $last_root_id = 0;

	public function register(): void {
		add_filter( 'wp_nav_menu_args', [ $this, 'inject_walker' ] );
	}

	/**
	 * Inject this walker only for 'footer' location.
	 */
	public function inject_walker( $args ) {
		if ( isset( $args['theme_location'] ) && $args['theme_location'] === 'footer' ) {
			$args['walker'] = $this;
		}
		return $args;
	}

	/**
	 * First pass: detect last top-level item, then use parent::walk().
	 */
	public function walk( $elements, $max_depth, ...$args ) {
		$root_ids = [];

		foreach ( $elements as $el ) {
			if ( (int) $el->menu_item_parent === 0 ) {
				$root_ids[] = (int) $el->ID;
			}
		}

		if ( ! empty( $root_ids ) ) {
			$this->last_root_id = (int) end( $root_ids );
		}

		return parent::walk( $elements, $max_depth, ...$args );
	}

	public function start_el( &$output, $item, $depth = 0, $args = null, $id = 0 ) {
		$title = esc_html( $item->title );
		$url   = esc_url( $item->url );

		// <li> classes
		$li_classes = [ 'menu-item', 'footer-menu-item' ];
		if ( 0 === $depth ) {
			$li_classes[] = 'd-flex';
			$li_classes[] = 'flex-column';
			$li_classes[] = 'paragraph';			
		}

		$output .= sprintf(
			'<li id="menu-item-%1$d" class="%2$s">',
			(int) $item->ID,
			esc_attr( implode( ' ', $li_classes ) )
		);

		$link_classes = '';
		if(0 === $depth) $link_classes = 'footer__title';
		// Basic link (можешь тут навесить свои классы по вкусу)
		$output .= '<a href="' . $url . '" class="'.$link_classes.'">' . $title . '</a>';

		/**
		 * IL content only for the last top-level item.
		 * 
		 * Берём контент из ACF options: поле 'footer_il_content'.
		 * Можно вставлять HTML, поэтому wp_kses_post.
		 */
		if ( 0 === $depth && (int) $item->ID === $this->last_root_id ) {

			$soc_list_title_footer = get_field( 'soc_list_title_footer', 'option' );
			$social_networks       = get_field( 'social_networks', 'option' );
		
			if ( $social_networks && is_array( $social_networks ) ) {
		
				$output .= '<div class="footer__soc">';
		
				if ( ! empty( $soc_list_title_footer ) ) {
					$output .= '<p class="footer__title paragraph">' . esc_html( $soc_list_title_footer ) . '</p>';
				}
		
				$output .= '<div class="nav__socList d-flex flex-wrap">';
		
				foreach ( $social_networks as $row ) {
					$logo = $row['logo'] ?? null;
					$link = $row['link'] ?? '';
		
					$output .= '<a ';
		
					if ( ! empty( $link ) ) {
						$output .= 'href="' . esc_url( $link ) . '" target="_blank" ';
					}
		
					$output .= 'class="nav__socList__item d-flex" tabindex="0">';
		
					// image render через CommonUtils
					$output .= \Headless\Utils\CommonUtils::render_image_acf(
						$logo,
						[
							'width'  => 32,
							'height' => 32,
							'alt'    => 'Soc icon',
						]
					);
		
					$output .= '</a>';
				}
		
				$output .= '</div>'; // .nav__socList
				$output .= '</div>'; // .footer__soc
			}
		}
		
		
	}

	public function end_el( &$output, $item, $depth = 0, $args = null ) {
		$output .= "</li>\n";
	}
}
