<?php
declare( strict_types=1 );

namespace Headless\Modules\Entities\Taxonomies;

use Headless\Modules\Entities\Traits\Labels;
use Headless\Interfaces\Registrable;
use Headless\Utils\CommonUtils;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class AbstractTaxonomy
 *
 * Base class for custom taxonomies
 * Provides label generation and default args for taxonomy registration
 */
abstract class AbstractTaxonomy implements Registrable {

	use Labels;

	/**
	 * Default arguments for post type registration
	 *
	 * @var array
	 */
	protected array $default_args = [
		'hierarchical'      => true,
		'show_ui'           => true,
		'show_admin_column' => true,
		'query_var'         => true,
		'show_in_graphql'   => true,
	];

	/**
	 * Get the taxonomy slug
	 *
	 * @return string
	 */
	abstract protected function get_slug(): string;

	/**
	 * Get the post types associated with the taxonomy
	 *
	 * @return array
	 */
	abstract protected function get_post_types(): array;

	/**
	 * Get the singular label
	 *
	 * @return string
	 */
	abstract protected function get_singular_label(): string;

	/**
	 * Get the plural label
	 *
	 * @return string
	 */
	abstract protected function get_plural_label(): string;

	/**
	 * Get the arguments for registering the taxonomy
	 *
	 * @return array
	 */
	abstract protected function get_custom_args(): array;

	/**
	 * Get the arguments for registering the taxonomy
	 *
	 * This method merges the default arguments with any custom arguments from the
	 * static get_custom_args() method. The labels are generated using the static taxonomy_labels()
	 * method
	 *
	 * @return array Arguments to use for registering the taxonomy
	 */
	protected function get_args(): array {

		// Set the default arguments
		$args = $this->default_args;

		// Args for graphql
		$args['show_in_graphql']     = true;
		$args['graphql_single_name'] = CommonUtils::snake_to_camel( $this->get_singular_label() );
		$args['graphql_plural_name'] = CommonUtils::snake_to_camel( $this->get_plural_label() );

		// Merge the default arguments with any custom arguments
		if ( ! empty( $this->get_custom_args() ) ) {
			$args = array_merge( $args, $this->get_custom_args() );
		}

		// Generate the labels for the taxonomy
		$args['labels'] = static::taxonomy_labels(
			$this->get_singular_label(),
			$this->get_plural_label()
		);

		return $args;
	}

	/**
	 * Registers the taxonomy with WordPress
	 *
	 * This method uses the get_args() method to generate the arguments for
	 * registering the taxonomy. The taxonomy slug is retrieved using the
	 * get_slug() method
	 *
	 * @return void
	 */
	public function register(): void {
		register_taxonomy( $this->get_slug(), $this->get_post_types(), $this->get_args() );
	}
}
