<?php
declare( strict_types=1 );

namespace Headless\Modules\Entities\PostTypes;

use Headless\Modules\Entities\Traits\Labels;
use Headless\Interfaces\Registrable;
use Headless\Utils\CommonUtils;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class AbstractPostType
 *
 * Base class for custom post types
 * Provides label generation and default args for CPT registration
 */
abstract class AbstractPostType implements Registrable {

	use Labels;

	/**
	 * Default arguments for post type registration
	 *
	 * @var array
	 */
	protected array $default_args = [
		'public'       => true,
		'supports'     => [ 'title', 'editor', 'thumbnail', 'revisions', 'excerpt', 'custom-fields' ],
		'has_archive'  => true,
		'show_in_rest' => true,
	];

	/**
	 * Get the post type slug
	 *
	 * @return string
	 */
	abstract protected function get_slug(): string;

	/**
	 * Get the singular label
	 *
	 * @return string
	 */
	abstract protected function get_singular_label(): string;

	/**
	 * Get the plural label
	 *
	 * @return string
	 */
	abstract protected function get_plural_label(): string;

	/**
	 * Return custom arguments for post type registration
	 *
	 * @return array Arguments to merge with the default arguments
	 */
	abstract protected function get_custom_args(): array;

	/**
	 * Get the arguments for register_post_type
	 *
	 * This method merges the default arguments with any custom arguments from the
	 * static args() method. The labels are generated using the static post_type_labels()
	 * method
	 *
	 * @return array Arguments to use for registering the post type
	 */
	protected function get_args(): array {

		// Set the default arguments
		$args            = $this->default_args;
		$args['rewrite'] = [ 'slug' => $this->get_slug() ];

		// Args for graphql
		$args['show_in_graphql']     = true;
		$args['graphql_single_name'] = CommonUtils::snake_to_camel( $this->get_singular_label() );
		$args['graphql_plural_name'] = CommonUtils::snake_to_camel( $this->get_plural_label() );

		// Merge the default arguments with any custom arguments
		if ( ! empty( $this->get_custom_args() ) ) {
			$args = array_merge( $args, $this->get_custom_args() );
		}

		// Generate the labels for the post type
		$args['labels'] = static::post_type_labels(
			$this->get_singular_label(),
			$this->get_plural_label()
		);

		return $args;
	}

	/**
	 * Registers the post type with WordPress
	 *
	 * This method uses the get_args() method to generate the arguments for
	 * registering the post type. The post type slug is retrieved using the
	 * get_slug() method
	 *
	 * @return void
	 */
	public function register(): void {
		register_post_type( $this->get_slug(), $this->get_args() );
	}

}
