<?php

declare( strict_types=1 );

namespace Headless\Modules\Blocks;

use Symfony\Component\Finder\Finder;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class BlockRegister
 *
 * Handles the registration of Gutenberg blocks in the WordPress theme
 */

class BlockRegister {
	/**
	 * Registers all Gutenberg blocks from specified directories
	 *
	 * This function iterates through the 'blocks' directory within the build path,
	 * identifies all JSON files which define block types, and registers each block type
	 *
	 * @return void
	 */
	public static function register_blocks(): void {

		// Get the current environment
		$env = defined( 'ENV' ) ? ENV : 'develop';

		// Get the cache key
		$cache_key = 'headless_block_files';

		// Check if the cache is valid
		if ( $env === 'production' ) {
			$blocks = get_transient( $cache_key );
			if ( $blocks === false ) {
				$blocks = static::scan_block_files();
				set_transient( $cache_key, $blocks, HOUR_IN_SECONDS );
			}
		} else {
			$blocks = static::scan_block_files();
		}

		// Register each block with the correct version
		foreach ( $blocks as $block_data ) {
			register_block_type( $block_data['path'] );
		}
	}

	/**
	 * Scans the block directory for JSON files that define block types
	 *
	 * @return array $blocks An array of block data
	 */
	private static function scan_block_files(): array {

		// Get the directory containing block JSON files
		$block_dir = static::get_block_dir();
		if ( ! $block_dir ) {
			return [];
		}

		// Initialize an array to hold block data
		$blocks = [];

		// Scan the block directory for JSON files
		$finder = new Finder();
		$finder->files()->in( $block_dir )->name( '*.json' );

		// Loop through each JSON file and add it to the blocks array
		foreach ( $finder as $file ) {
			$blocks[] = [ 'path' => $file->getRealPath() ];
		}

		return $blocks;

	}


	/**
	 * Retrieves the directory containing the block JSON files
	 *
	 * @return string|bool Returns the directory path if it exists, or false if it doesn't
	 */
	private static function get_block_dir(): string|bool {

		// Check if the BUILD_PATH constant is defined
		if ( ! defined( 'BUILD_PATH' ) ) {
			error_log( 'BlocksManager: BUILD_PATH is not defined.' );

			return false;
		}

		// Define the directory containing block JSON files
		$block_dir = BUILD_PATH . trailingslashit( 'blocks' );

		// Check if the block directory exists
		if ( ! is_dir( $block_dir ) ) {
			error_log( "BlocksManager: Directory $block_dir does not exist" );

			return false;
		}

		return $block_dir;
	}
}